import os
import subprocess
import yaml
from pathlib import Path
import sys

CONFIG = Path("settings.yaml")

DEFAULT_CONFIG = {
    "connection": {"name": "Anthro-Windows-VPS"},
    "ssh": {
        "host": "YOUR_VPS_IP",
        "port": 22,
        "user": "sshuser",
        "key": "~/.ssh/id_rsa",
    },
    "rdp": {
        "windows_host": "127.0.0.1",
        "local_port": 3390,
        "user": "Administrator",
        "password": "CHANGE_ME",
        "start_resolution": "1024x768",
        "fullscreen_resolution": "1920x1080",
    },
    "performance": {
        "network_profile": "lan",
        "compression": False,
        "gfx": True,
    },
}

def load_or_create_config():
    if not CONFIG.exists():
        with open(CONFIG, "w") as f:
            yaml.safe_dump(DEFAULT_CONFIG, f)
        print("settings.yaml created. Edit it, then rerun.")
        sys.exit(0)

    with open(CONFIG) as f:
        return yaml.safe_load(f)

def start_ssh_tunnel(cfg):
    ssh = cfg["ssh"]
    rdp = cfg["rdp"]

    cmd = [
        "ssh",
        "-N",
        "-L",
        f"{rdp['local_port']}:localhost:3389",
        "-i",
        os.path.expanduser(ssh["key"]),
        f"{ssh['user']}@{ssh['host']}",
    ]

    return subprocess.Popen(cmd)

def start_rdp(cfg):
    rdp = cfg["rdp"]
    perf = cfg["performance"]

    cmd = [
        "xfreerdp",
        f"/v:{rdp['windows_host']}:{rdp['local_port']}",
        f"/u:{rdp['user']}",
        f"/p:{rdp['password']}",
        f"/size:{rdp['start_resolution']}",
        "/dynamic-resolution",
        "/audio-mode:0",
        "/microphone",
        "/usb:auto",
        f"/network:{perf['network_profile']}",
        "/compression-level:0" if not perf["compression"] else "",
        "/gfx" if perf["gfx"] else "",
        "/cert:ignore",
    ]

    subprocess.run([c for c in cmd if c])

def main():
    cfg = load_or_create_config()
    print(f"Connecting to {cfg['connection']['name']}")

    tunnel = start_ssh_tunnel(cfg)

    try:
        start_rdp(cfg)
    finally:
        tunnel.terminate()

if __name__ == "__main__":
    main()
